import { Request, Response } from 'express';
import { debugService } from './service';

export const debugController = {
  // GET /debug/ping
  async ping(req: Request, res: Response) {
    const data = await debugService.ping();
    return res.json({ success: true, data });
  },

  // GET /debug/supabase
  async testSupabase(req: Request, res: Response) {
    const data = await debugService.testSupabaseConnection();
    return res.json({ success: true, data });
  },

  // GET /debug/user/:user_id
  async getUserDebugInfo(req: Request, res: Response) {
    const { user_id } = req.params;
    const { data, error } = await debugService.getUserDebugInfo(user_id);
    if (error) return res.status(400).json({ error });
    return res.json({ success: true, data });
  },

  // POST /debug/feature-check
  async checkFeature(req: Request, res: Response) {
    const { user_id, feature } = req.body;
    if (!user_id || !feature) {
      return res.status(400).json({ error: 'user_id and feature are required' });
    }
    
    const { data, error } = await debugService.checkFeature({ user_id, feature });
    if (error) return res.status(400).json({ error });
    return res.json({ success: true, data });
  },

  // POST /debug/subscription-check
  async checkSubscription(req: Request, res: Response) {
    const { vendor_id } = req.body;
    if (!vendor_id) {
      return res.status(400).json({ error: 'vendor_id is required' });
    }
    
    const { data, error } = await debugService.checkSubscription({ vendor_id });
    if (error) return res.status(400).json({ error });
    return res.json({ success: true, data });
  },

  // POST /debug/ai-log
  async testAILog(req: Request, res: Response) {
    const { message, source_module, user_id } = req.body;
    if (!message || !source_module) {
      return res.status(400).json({ error: 'message and source_module are required' });
    }
    
    const { data, error } = await debugService.testAILog({ message, source_module, user_id });
    if (error) return res.status(400).json({ error });
    return res.json({ success: true, data });
  },

  // GET /debug/system-status
  async getSystemStatus(req: Request, res: Response) {
    const { data, error } = await debugService.getSystemStatus();
    if (error) return res.status(400).json({ error });
    return res.json({ success: true, data });
  },

  // GET /debug/cache
  async getCacheDebugInfo(req: Request, res: Response) {
    const { user_id } = req.query;
    if (!user_id || typeof user_id !== 'string') {
      return res.status(400).json({ error: 'user_id query parameter is required' });
    }
    
    const { data, error } = await debugService.getCacheDebugInfo(user_id);
    if (error) return res.status(400).json({ error });
    return res.json({ success: true, data });
  }
}; 